#include <avr/io.h>
#include <avr/interrupt.h>

// Geblinkt wird PortB.1 (push-pull)
// Eine LED in Reihe mit einem Vorwiderstand zwischen
// PortB.1 und GND anschliessen.
#define PAD_LED  1
#define PORT_LED PORTB
#define DDR_LED  DDRB

// Der MCU-Takt. Wird gebraucht, um Timer1 mit den richtigen
// Werten zu initialisieren. Voreinstellung ist 1MHz.
// (Werkseinstellung fr AVRs mit internem Oszillator).
// Das Define wird nur gemacht, wenn F_CPU noch nicht definiert wurde.
// F_CPU kann man so auch per Kommandozeile definieren, z.B. fr 8MHz:
// avr-gcc ... -DF_CPU=8000000
//   
// ! Der Wert von F_CPU hat rein informativen Character fr
// ! die korrekte Codeerzeugung im Programm!
// ! Um die Taktrate zu ndern, mssen die Fuses des Controllers
// ! und/oder Quarz/Resonator/RC-Glied/Oszillator angepasst werden!
#ifndef F_CPU
#define F_CPU    1000000
#endif

// So viele IRQs werden jede Sekunde ausgelst.
// Fr optimale Genauigkeit muss IRQS_PER_SECOND ein Teiler
// von F_CPU sein und IRQS_PER_SECOND ein Vielfaches von 100.
// Ausserdem muss gelten F_CPU / IRQS_PER_SECOND <= 65536
#define IRQS_PER_SECOND   2000 /* 500 s */

// Anzahl IRQs pro 10 Millisekunden
#define IRQS_PER_10MS     (IRQS_PER_SECOND / 100)

// Gltigkeitsprfung.
// Bei ungeeigneten Werten gibt es einen Compilerfehler
#if (F_CPU/IRQS_PER_SECOND > 65536) \
   || (IRQS_PER_10MS < 1)           \
   || (IRQS_PER_10MS > 255)
#   error Diese Werte fuer F_CPU und IRQS_PER_SECOND
#   error sind ausserhalb des gueltigen Bereichs!
#endif

// Compiler-Warnung falls die Genauigkeit nicht optimal ist.
// Wenn das nervt fr deine Werte, einfach lschen :-)
#if (F_CPU % IRQS_PER_SECOND != 0) || (IRQS_PER_SECOND % 100 != 0)
#   warning Das Programm arbeitet nicht mit optimaler Genauigkeit.
#endif

// Prototypen
void wait_10ms (const uint8_t);
void timer1_init();

// Zhler-Variable. Wird in der ISR erniedrigt und in wait_10ms benutzt.
static volatile uint8_t timer_10ms;

// //////////////////////////////////////////////////////////////////////
// Implementierungen der Funktionen
// //////////////////////////////////////////////////////////////////////

// //////////////////////////////////////////////////////////////////////
// Timer1 so initialisieren, da er IRQS_PER_SECOND 
// IRQs pro Sekunde erzeugt.
#if !defined (TCNT1H)
#error Dieser Controller hat keinen 16-Bit Timer1!
#endif // TCNT1H
void timer1_init()
{
    // Timer1: keine PWM
    TCCR1A = 0;

    // Timer1 ist Zhler: Clear Timer on Compare Match (CTC, Mode #4)
    // Timer1 luft mit vollem MCU-Takt: Prescale = 1
#if defined (CTC1) && !defined (WGM12)
    TCCR1B = (1 << CTC1)  | (1 << CS10);
#elif !defined (CTC1) && defined (WGM12)
    TCCR1B = (1 << WGM12) | (1 << CS10);
#else
#error Keine Ahnung, wie Timer1 fuer diesen Controller zu initialisieren ist!
#endif

    // OutputCompare fr gewnschte Timer1 Frequenz
    // TCNT1 zhlt immer 0...OCR1A, 0...OCR1A, ... 
    // Beim berlauf OCR1A -> OCR1A+1 wird TCNT1=0 gesetzt und im nchsten
    // MCU-Takt eine IRQ erzeugt.
    OCR1A = (unsigned short) ((unsigned long) F_CPU / IRQS_PER_SECOND-1);

    // OutputCompareA-Interrupt fr Timer1 aktivieren
#if defined (TIMSK1)
    TIMSK1 |= (1 << OCIE1A);
#elif defined (TIMSK)
    TIMSK  |= (1 << OCIE1A);
#else	 
#error Keine Ahnung, wie IRQs fuer diesen Controller zu initialisieren sind!
#endif
}

// //////////////////////////////////////////////////////////////////////
// Wartet etwa t*10 ms mit 0 < t < 255. 
// timer_10ms wird alle 10ms in der Timer1-ISR erniedrigt. 
// Weil es bis zum nchsten IRQ nicht lnger als 10ms dauert,
// wartet diese Funktion zwischen t*10 ms und (t+1)*10 ms.
void wait_10ms (const uint8_t t)
{
    timer_10ms = t+1;
    while (timer_10ms);
}

// //////////////////////////////////////////////////////////////////////
// Die Interrupt Service Routine (ISR).
// In interrupt_num_10ms werden die IRQs gezhlt.
// Sind IRQS_PER_10MS Interrups geschehen, dann sind 10 ms vergangen.
// timer_10ms wird alle 10 ms um 1 vermindert und bleibt bei 0 stehen.
SIGNAL (SIG_OUTPUT_COMPARE1A)
{
    static uint8_t interrupt_num_10ms;

    // interrupt_num_10ms erhhen und mit Maximalwert vergleichen
    if (++interrupt_num_10ms == IRQS_PER_10MS)
    {
        // 10 Millisekunden sind vorbei
        // interrupt_num_10ms zurcksetzen
        interrupt_num_10ms = 0;

        // Alle 10ms wird timer_10ms erniedrigt, falls es nicht schon 0 ist.
        // Wird verwendet in wait_10ms
        if (timer_10ms != 0)
            timer_10ms--;
    }
}

// //////////////////////////////////////////////////////////////////////
// Das Hauptprogramm: Startpunkt 
int main()
{
    // LED-Port auf OUT
    DDR_LED  |= (1 << PAD_LED);

    // Timer1 initialisieren
    timer1_init();

    // Interrupts global aktivieren
    sei();

    // Endlosschleife
    // Die LED ist jeweils 1 Sekunde an und 1 Sekunde aus,
    // blinkt also mit einer Frequenz von 0.5 Hz
    while (1)
    {
        // LED an
        PORT_LED |= (1 << PAD_LED);

        // 1 Sekunde warten
        wait_10ms (100);

        // LED aus
        PORT_LED &= ~(1 << PAD_LED);

        // 1 Sekunde warten
        wait_10ms (100);
    }

    // main braucht keine return-Anweisung, weil wir nie hier hin kommen
}
